/**
 * 创建一个播放器
 * @param {*} obj 输入参数对像
 */
function SLPlayer(obj) {
	return new Promise(function (resolve) {
		resolve(new SLVideo(obj));
	});
}

/**
 * 创建AudioContext对像
 */
function SLAudioContext() {
	return new (window.AudioContext || window.webkitAudioContext)();
}

/**
 * 队列数据
 */
class SLLoopCache {
	constructor(size) {
		this._cache = new Float32Array(size);
		this._start = 0;
		this._end = 0;
	}

	// 将采集的音视频入栈
	push(buff) {
		if (this._end >= this._start) {
			// 向后导入数据
			if ((this._end + buff.length) <= this._cache.length) {
				// 空间充足
				this._cache.set(buff, this._end);
				this._end += buff.length;
				if (this._end >= this._cache.length) {
					this._end = 0;
					// 如果开始位置没有动，就向前移
					if (0 == this._start) {
						this._start += buff.length;
					}
				}
			} else {
				// 空间不足的情况下
				var s = this._cache.length - this._end;
				var a = buff.subarray(0, s);
				var b = buff.subarray(s);
				// 分两次入栈
				this.push(a);
				this.push(b);
			}
		} else {
			// 带删除功能导入
			if ((this._end + buff.length) < this._start) {
				// 空间充足
				this._cache.set(buff, this._end);
				this._end += buff.length;
			} else {
				// 空间不足时删除空间
				this._start += buff.length;
				if (this._start >= this._cache.length) {
					this._start = 0;
				}

				this.push(buff);
			}
		}
	}

	// 获取缓冲区大小
	size() {
		if (this._start == this._end) {
			return 0;
		} else if (this._end > this._start) {
			return this._end - this._start;
		} else {
			return this._end + this._cache.length - this._start;
		}
	}

	// 消息出栈
	pop(size) {
		if (this.size() < size) {
			return null;
		}

		var ret = new Float32Array(size);
		if (this._end > this._start) {
			// 正常删除
			ret.set(this._cache.subarray(this._start, this._start + size), 0);
			this._start += size;
		} else {
			// 分段删除
			if ((this._start + size) <= this._cache.length) {
				// 未分段
				ret.set(this._cache.subarray(this._start, this._start + size), 0);
				this._start += size;
				if (this._start >= this._cache.length) {
					this._start = 0;
				}
			} else {
				// 已分段
				var s = this._cache.length - this._start;
				// 第一段
				ret.set(this._cache.subarray(this._start), 0);
				// 第二段
				ret.set(this._cache.subarray(0, size - s), s);
				this._start = size - s;
			}
		}

		return ret;
	}
}

/**
 * 声音播放类
 */
class SLAudioPlayer {
	constructor() {
		this._context = null;
		this._state = 0;
	}

	/**
	 * 开始播放声音
	 */
	start() {
		if (null == this._context) {
			this._context = SLAudioContext();
			this._nextTime = 0;
			// 播放一个字节的数据
			var pcm = new Float32Array(1280);
			this.play(1, 32000, pcm);
		}
	}

	/**
	 * 停止播放声音
	 */
	stop() {
		if (null != this._context) {
			this._context.close();
			this._context = null;
		}
	}

	/**
	 * 控制声音
	 * @param {*} v 
	 */
	ctrl(v) {
		this._state = v;
		// 判断是否开启声音
		if (0 != this._state && null == this._context) {
			this.start();
		}
	}

	/**
	 * 停止播放声音
	 */
	play(channel, sample, pcm) {
		// 判断是否创建声音对像
		if (null == this._context || 0 == this._state) {
			return;
		}

		// 构建播放数据
		var buff = this._context.createBuffer(channel, pcm.length, sample);
		buff.getChannelData(0).set(pcm, 0);
		// 创建播放源
		var source = this._context.createBufferSource();
		source.buffer = buff;
		source.connect(this._context.destination);
		if (this._nextTime < this._context.currentTime) {
			this._nextTime = this._context.currentTime;
		}

		source.start(this._nextTime);
		this._nextTime += source.buffer.duration;
	}
}

/**
 * 执行声音采集
 */
class SLAudioCapture {
	constructor(callback) {
		this._callback = callback;
		this._context = null;
		this._captrue = 0;
	}

	/**
	 * 开始采集
	 */
	start() {
		if (null != this._context) {
			return;
		}

		// 创建3秒的缓存数据区间
		if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
			// 开始创建录音
			console.log('getUserMedia supported.');
			// 执行创建流程
			navigator.mediaDevices.getUserMedia({ audio: true })
				// 执行数据回调函数
				.then((stream) => { this._onCaptureStart(stream); })
				// Error callback
				.catch(function (err) {
					console.error('The following getUserMedia error occured: ' + err);
				});
		} else {
			console.error("getUserMedia not supported.");
		}
	}

	/**
	 * 停止采集
	 */
	stop() {
		if (null != this._context) {
			// 断开采集
			this._input.disconnect(this._processor);
			this._processor.disconnect(this._context.destination);
			// 释放对像
			this._context.close();
			this._context = null;
		}
	}

	/**
	 * 控制采集
	 */
	ctrl(v) {
		this._state = v;
		if (0 != this._state && null == this._context) {
			this.start();
		}
	}

	/**
	 * 开始采集数据
	 */
	_onCaptureStart(stream) {
		this._context = SLAudioContext();
		// 创建3秒的缓冲区
		this._queue = new SLLoopCache(this._context.sampleRate * 3);
		// 创建接收pcm数据区
		this._input = this._context.createMediaStreamSource(stream)
		this._processor = this._context.createScriptProcessor(1024, 1, 1);

		this._input.connect(this._processor);
		this._processor.connect(this._context.destination);
		// 设置数据回调函数
		this._processor.onaudioprocess = (e) => { this._onCaptureData(e.inputBuffer); };
	}

	/**
	 * 数据回调函数
	 */
	_onCaptureData(data) {
		// 判断是否正在录音
		if (0 == this._state) {
			return;
		}

		var buff = data.getChannelData(0);
		// 将数据加入到缓存中
		this._queue.push(buff);
		// 取数据，然后发送出去
		while (true) {
			var data = this._queue.pop(this._context.sampleRate / 25);
			if (null == data) {
				break;
			}

			// 调用回调函数
			if ("function" == typeof (this._callback)) {
				this._callback(this._context.sampleRate, data);
			}
		}
	}
}

/**
 * 播放器类
 */
class SLVideo {
	/**
	 * 
	 * @param {*} param 播放器类需要的参数
	 */
	constructor(param) {

		// 将参数复制过去
		for (var a in param) {
			this[a] = param[a];
		}

		// 初始化参数
		if (typeof (this.canvas) == "object" && null != this.canvas) {
			this._dc = this.canvas.getContext("2d");
		} else {
			this._dc = null;
		}

		this._worker = null;
		this._ratio = window.devicePixelRatio || 1;
		this._audioPlayer = new SLAudioPlayer();
		this._audioCapture = new SLAudioCapture((sample, pcm) => { this._onCapturePCM(sample, pcm) });
	}

	/**
	 * 打印版本号
	 */
	version() {
		return "v5.2.0(20191115)";
	}

	/**
	 * 开始播放
	 * @param {*} uri 播放的地址
	 */
	start(uri) {
		// 先前闭之前的连接
		this.stop();

		// 将视频显示为黑色
		this._showBlack();
		// 初始化环境变量
		this._hasVideo = false;

		// 保存uri
		this._uri = uri;
		// 创建新的工作线程
		this._worker = new Worker(this.decoder || "./sldecoder.js");
		this._worker.onmessage = (e) => { this._onWorkerMessage(e); }
		this._worker.onerror = (e) => { this._onWorkerError(e); }

		// 判断是否为对讲模式
		if (null != this._dc) {
			return;
		}

		// 播放声音
		this._audioPlayer.ctrl(1);
	}

	/**
	 * 停止播放
	 */
	stop() {
		if (null == this._worker) {
			return;
		}

		// 停止声音播入和采集
		this._audioPlayer.stop();
		this._audioCapture.stop();
		// 停止工作线程
		this._worker.terminate();
		this._worker = null;
		// 调用退出
		if ("function" == typeof (this.onEnd)) {
			this.onEnd();
		}

		// 显示为黑屏
		this._showBlack();
	}

	/**
	 * 截图
	 */
	capture() {
		return this.canvas.toDataURL("image/jpeg");
	}

	/**
	 * 播放声音
	 * @param {*} ctrl 0=表示停止播放, 1=表示开始播放
	 */
	voice(ctrl) {
		if (0 == ctrl) {
			// 关闭声音
			this._audioPlayer.ctrl(0);
			if (null == this._dc) {
				this._audioCapture.ctrl(1);
			}
		} else {
			// 开启声音
			this._audioPlayer.ctrl(1);
			if (null == this._dc) {
				this._audioCapture.ctrl(0);
			}
		}
	}

	_showBlack() {
		if (null == this._dc) {
			return;
		}

		this._dc.fillStyle = "black";
		this._dc.fillRect(0, 0, this.canvas.width, this.canvas.height);
	}

	/**
	 * 输出错误
	 * @param {*} code 错误代码
	 * @param {*} err 错误提示
	 */
	_error(code, err) {
		// 发生错误后，需要调用停止
		this.stop();
		// 输出错误
		var str;
		if ("object" == typeof (err) && "string" == typeof (err.message)) {
			str = err.message;
		}
		else {
			str = String(err);
		}

		if ("function" == typeof (this.onError)) {
			this.onError(code, str);
		} else {
			console.error(code + ":  " + str);
		}
	}

	/**
	 * 打印日志
	 * @param {*} l 
	 */
	_log(l) {
		if ("function" == typeof (this.print)) {
			this.print(l);
		} else {
			console.log(l);
		}
	}

	/**
	 * 线程消息
	 * @param {*} event
	 */
	_onWorkerMessage(event) {
		// 判断数据是否为对像
		if ("object" != typeof (event.data)) {
			console.error("worker response error: event.data is not object");
			return;
		}

		// 判断相应的参数
		if ("string" != typeof (event.data.m)) {
			console.error("worker response error: event.data.m is not undefined.");
			return;
		}

		// 进行相应的函数调度
		switch (event.data.m) {
			case "onWorkerInit": this._onWorkerInit(event.data); break;
			case "onWorkerConnect": this._onWorkerConnect(event.data); break;
			case "onWorkerError": this._onWorkerError(event.data); break;
			case "onWorkerRGBA": this._onWorkerRGBA(event.data); break;
			case "onWorkerPCM": this._onWorkerPCM(event.data); break;
			default: console.error("can not found method: ", event.data.m); break;
		}
	}

	/**
	 * 线程错误消息
	 * @param {*} err 
	 */
	_onWorkerError(err) {
		this._error(2, "start worker error");
	}

	/**
	 * 线程初始化
	 * @param {*} obj 
	 */
	_onWorkerInit(obj) {
		this._worker.postMessage({ "m": "onWorkerConnect", "url": this._uri });
	}

	/**
	 * 连接成功回调事件
	 * @param {*} obj 
	 */
	_onWorkerConnect(obj) {
		if ("function" == typeof (this.onStart)) {
			this.onStart();
		}
	}

	_onWorkerError(obj) {
		this._error(obj.c, obj.e);
	}

	/**
	 * 绘制RGBA
	 * @param {*} obj 
	 */
	_onWorkerRGBA(obj) {
		if (null == this._dc) {
			console.error("can't find canvas");
			return;
		}

		// 重置画面大小
		if (obj.w != this.canvas.width || obj.h != this.canvas.height) {
			console.debug("w=" + obj.w + "; h=" + obj.h);
			this.canvas.width = obj.w;
			this.canvas.height = obj.h;
		}

		// 获取数据区
		var dst = this._dc.getImageData(0, 0, obj.w, obj.h);
		var src = new Uint8Array(obj.rgba);
		// 复制数据
		if (src.length <= dst.data.length) {
			dst.data.set(src, 0);
		} else {
			// console.debug("video is lage: ", dst.data.length, "<>", src.length);
			dst.data.set(src.subarray(0, dst.data.length), 0);
		}

		// 将数组退还给工作线程
		this._doWorkerACK("rgba", obj.rgba);
		// 显示到画面
		this._dc.putImageData(dst, 0, 0);

		this.flow = obj.flow;
		// 输出时间戳
		if ("function" == typeof (this.onPlayTime)) {
			if (obj.ts !== this._lastPlayTime) {
				this._lastPlayTime = obj.ts;
				this.onPlayTime(obj.ts, obj.flow);
			}
		}

		// 事件回调
		if (true !== this._hasVideo) {
			this._hasVideo = true;
			if ("function" == typeof (this.onVideoStart)) {
				this.onVideoStart();
			}
		}
	}

	// 声音数据回调
	_onWorkerPCM(obj) {
		// {m: "onWorkerPCM", s: 8000, b: 16, c: 1, pcm: ArrayBuffer(640)}
		var pcm = new Float32Array(obj.pcm);
		this._audioPlayer.play(obj.c, obj.s, pcm);
	}

	/**
	 * 回复工作线程数据
	 * @param {*} name 
	 * @param {*} value 
	 */
	_doWorkerACK(name, value) {
		this._worker.postMessage({
			"m": "onWorkerACK",
			"name": name,
			"value": value
		}, [value])
	}

	/**
	 * 发送数据
	 * @param {*} sample 
	 * @param {*} pcm 
	 */
	_onCapturePCM(sample, pcm) {
		if (this._worker && this._worker.postMessage) {
			// 发送pcm数据
			this._worker.postMessage({
				"m": "onWorkerTalk",
				"s": sample,
				"pcm": pcm.buffer,
			}, [pcm.buffer]);
		}
	}
};

